
; Some basic routines to drive an LCD graphic module in both graphic and
; alpha modes. A lot of this file is graphic data for four pictures that
; form a small slideshow loop. Interestingly the data is run length limit
; encoded and is unpacked, on the fly, by the Unpk_2_LCD routine.

; The display plugs into a decoded port on the uPF main board, the -ve
; display drive voltage was generated using a MAX232 chip.

; If you want to ROM this code you will need to make sure the variables
; are in RAM and write a short bit of code to initialise them.

; Interface equates. Note C2h and C3h are read ONLY.

LCD_wrdata	EQU	C0h	; write data
LCD_wrinst	EQU	C1h	; write instruction
LCD_rddata	EQU	C2h	; read data
LCD_rdbusy	EQU	C3h	; read busy flag (and address)

; Display attribute equates

DispOn:		EQU	20h	; display on bit
DispBlnk:	EQU	08h	; cursor blink bit
DispCrsr:	EQU	04h	; cursor on bit
DispGrap:	EQU	02h	; display graph mode bit
DispExCg:	EQU	01h	; external GC bit

	ORG	8000h		; start in RAM
Start:
	CALL	LCD_init	; set initial values

	LD	A,#08h		; new chr width
	CALL	LCD_chrw	; go set it

	LD	A,#08h		; display on
	CALL	LCD_setchrmode	; set display mode

	CALL	LCD_clr		; clear the display	

; Display a message on the LCD

	LD	HL,Msg		; point to message
	LD	BC,MsgEnd-Msg	; mesage length	
	CALL	WriteL		; write the message

	CALL	Wait		; delay

	LD	A,#07h		; new chr width
	CALL	LCD_chrw	; go set it

	LD	BC,0047h	; set cursor
	CALL	Set_cursor	; set the cursor
	LD	HL,Fix36	; point to message
	LD	BC,Fix42-Fix36	; mesage length	
	CALL	WriteL		; write the message

	CALL	Wait		; delay

	LD	A,#06h		; new chr width
	CALL	LCD_chrw	; go set it

	LD	BC,0046h	; set cursor
	CALL	Set_cursor	; set the cursor
	LD	HL,Fix42	; point to message
	LD	BC,FixEn-Fix42	; mesage length	
	CALL	WriteL		; write the message

	CALL	Wait		; delay

	XOR	A		; display off
	CALL	LCD_setgramode	; set graph mode

	CALL	LCD_clr		; clear the display	
	CALL	LCD_on		; turn it on

	LD	HL,Graphics	; point to data
	LD	B,(HL)		; get # of images to do
	INC	HL		; point to first count
Flash:
	LD	E,(HL)		; get count low
	INC	HL		; point to high
	LD	D,(HL)		; get count high
	INC	HL		; point to data

	PUSH	BC		; save image count

	CALL	Unpk_2_LCD	; unpack compressed data to LCD	

	CALL	Wait		; delay

	POP	BC		; get count back
	DJNZ	Flash		; display next pic

	JP	Start		; go to start
;	HALT			; end of code

; Writes the message at (HL) of length BC to the display

WriteL:
	LD	A,(HL)		; get byte
	CALL	LCD_print	; out to LCD
	INC	HL		; point to next chr
	DEC	BC		; decrement count
	LD	A,B		; get high byte
	OR	C		; OR in low byte
	JR	NZ,WriteL	; loop if not all done
	RET			; done

Wait:
	LD	A,#4		; loop 4 times
	LD	BC,0000		; long wait
Wait2:
	DJNZ	Wait2		; loop
	DEC	C		; inner loop
	JR	NZ,Wait2	; loop
	DEC	A		; decr loop count
	JR	NZ,Wait2	; loop
	RET			; done

; Start with HL pointing to first byte of compressed data and DE
; holding the byte count.

Unpk_2_LCD:
	LD	A,#0Ch		; write display data
	CALL	LCD_instr	; set instruction
UnpkStart:
	LD	A,(HL)		; get count/flag byte
	DEC	DE		; decrement count
	INC	HL		; increment pointer

	LD	C,A		; save count/flag byte
	AND	#7Fh		; mask count
	LD	B,A		; save it
	BIT	7,C		; test "same" flag bit
	JR	Z,ExtrDiff	; extract diff string
				; unpacks B bytes of same data
	LD	A,(HL)		; get byte
	DEC	DE		; decrement count
	INC	HL		; increment pointer
SameOut:
	CALL	LCD_busy	; wait until ready
	OUT	(LCD_wrdata),A	; chr out to display
	DJNZ	SameOut		; loop until bytes written
UnpkEnd:
	LD	A,D		; get high byte
	OR	E		; or low byte
	JR	NZ,UnpkStart	; do next if not all done
	RET
ExtrDiff:			; extracts B bytes of diff data
	LD	A,(HL)		; get byte
	DEC	DE		; decrement count
	INC	HL		; increment pointer
	CALL	LCD_busy	; wait until ready
	OUT	(LCD_wrdata),A	; chr out to display
	DJNZ	ExtrDiff	; loop until bytes written
	JR	UnpkEnd		; done so go back

; Clear LCD display. Turns off display, checks mode and then fills 2048
; display bytes with either 00h (graph mode) or 20h (chr mode). It also
; homes the cursor.

; If the commented out instructions are restored then the display will be
; blanked during clearing.

LCD_clr:
	LD	A,(DispMode)	; get mode
;	AND	#1Fh		; display off
;	CALL	LCD_mode	; set display mode
	AND	#02h		; check chr/graph mode
	LD	A,#00h		; clear A (don't set flags!)
	LD	HL,CursorAt	; get pointer to cursor pos'n
	LD	(HL),A		; clear byte
	INC	HL		; next byte
	LD	(HL),A		; clear byte
	JR	NZ,ClGraph	; go clear graph mode
	LD	A,#20h		; space chr to clear chr mode display
ClGraph:
	PUSH	AF		; save it
	LD	BC,0000h	; reset cursor
	CALL	Set_cursor	; set cursor position
	LD	C,#08h		; 8 x 256 = 2048 bytes
	LD	A,#0Ch		; write display data
	CALL	LCD_instr	; set instruction
	POP	AF		; get byte back
Clear_l:
	CALL	LCD_busy	; wait until ready
	OUT	(LCD_wrdata),A	; chr out to display
	DJNZ	Clear_l		; loop until all chrs done
	DEC	C		; inner loop
	JR	NZ,Clear_l	; loop if not all done
	CALL	Set_cursor	; set cursor position (BC will be 0000)
;	LD	A,(DispMode)	; get mode
;	CALL	LCD_mode	; set display mode
	RET			; done

; Set cursor address. 16 bit value held in BC

Set_cursor:
	PUSH	AF		; save AF

	LD	A,#0Ah		; Set cursor low addr instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,C		; set low address
	OUT	(LCD_wrdata),A	; set mode

	LD	A,#0Bh		; Set cursor high addr instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,B		; set high address
	OUT	(LCD_wrdata),A	; set mode

	LD	(CursorAt),BC	; save cursor position

	POP	AF		; restore AF
	RET

; Need to check if the cursor is inside the visible window. If it isn't I
; should then scroll the window left/right (in TAB size steps) or up/down
; until it is....... but first of all check that autoscroll is on!

;ScrlChk:
;	LD	A,(AutoScr)	; get autoscroll flag
;	OR	A		; set the flags
;	RET	Z		; exit if zero
;	LD	A,(DispMode)	; get mode
;	AND	#02h		; check chr/graph mode
;	RET	NZ		; exit if graph mode
;
;	RET


; set display start address

Set_start:
	PUSH	AF		; save AF

	LD	A,#08h		; Set display start low addr instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,C		; set low byte
	OUT	(LCD_wrdata),A	; set mode

	LD	A,#09h		; Set display start high addr instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,B		; set high byte
	OUT	(LCD_wrdata),A	; set mode

	LD	(StrtOff),BC	; save start offset

	POP	AF		; restore AF
	RET

; Initialise 256x64 LCD display. Sets line length to 64 and time divisions
; to 64

LCD_init:
	LD	A,#03h		; Set time divisions instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,#40h		; set to 64
	OUT	(LCD_wrdata),A	; set mode
	RET
; Turn on the LCD. without changing the mode parameters.

LCD_on:
	LD	A,(DispMode)	; get the display mode
	OR	#DispOn		; turn on bit
	LD	(DispMode),A	; save mode
	JP	LCD_mode	; set display mode & return
;	RET

; Turn off the LCD. without changing the mode parameters.

LCD_off:
	LD	A,(DispMode)	; get the display mode
	AND	#FFh-DispOn	; turn off bit
	LD	(DispMode),A	; save mode
;	JP	LCD_mode	; set display mode & return
;	RET

; Set the LCD mode. mode byte should be in A

LCD_mode:
	PUSH	AF		; save mode byte
	XOR	A		; clear A
	CALL	LCD_instr	; set display control instruction
	POP	AF		; get mode byte back
	OUT	(LCD_wrdata),A	; set mode
	RET

; Output instruction. We need to wait until the busy flag is clear before
; attempting to write a new instruction.

LCD_instr:			; wait & output instruction
	CALL	LCD_busy	; wait until ready
	OUT	(LCD_wrinst),A	; out to display
	RET			; done

; Wait until the display is not busy. Preserves A.

LCD_busy:			; wait until ready
	PUSH	AF		; save AF
LCD_busy1:
	IN	A,(LCD_rdbusy)	; get status (& address)
	BIT	7,A		; check status bit
	JR	NZ,LCD_busy1	; loop if still busy
	POP	AF		; restore AF
	RET			; done

; Set graph mode. This will set the width to eight and the display to 32
; chrs/line. If you want the display on when graph mode is set call this
; routine with A<>0 .

LCD_setgramode:
	OR	A		; set the falgs
	LD	A,#12h		; display off/master/graph mode
	JR	Z,GsetOff	; jump if display is to be off
	LD	A,#32h		; display on/master/graph mode
GsetOff:
	LD	(DispMode),A	; save mode
	CALL	LCD_mode	; set display mode

	LD	A,#01h		; set chr pitch instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,#07h		; chr width 8 bits/chr
	OUT	(LCD_wrdata),A	; set mode

	LD	A,#02h		; set no. of chrs instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,#1Eh		; 32 chr mode
	OUT	(LCD_wrdata),A	; set mode

	LD	BC,0000h	; reset start of display
	JP	Set_start	; set start address & return
;	RET

; Set character mode. This will read the default values and use those for
; the chr width and height, it will also set the display to 64 chrs/line.
; If you want the display on when chr mode is set call this routine with
; A<>0 .

LCD_setchrmode:
	OR	A		; set the falgs
	LD	A,#10h		; display off/master/chr mode
	JR	Z,CsetOff	; jump if display is to be off
	LD	A,#30h		; display on/master/chr mode
CsetOff:
	LD	(DispMode),A	; save mode
	CALL	LCD_mode	; set display mode

	LD	A,#02h		; Set no. of chrs instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,#3Eh		; 64 chr mode
	OUT	(LCD_wrdata),A	; set mode

	LD	A,(ChrWidt)	; get chr width
	INC	A		; +1 for routine
;	JP	LCD_chrw	; go set it & return
;	RET

; Set the character width for the chr mode display. Width is in A, valid
; widths are 6,7 and 8 bits. After the width is set the new visible length
; and the new start offset are set.

LCD_chrw:
	SUB	#06h		; convert to 0,1,2
	RET	C		; exit if was less than 6
	CP	#03h		; compare with 9
	RET	NC		; exit if not less (range 6 to 8)
	ADD	A,A		; *2
	ADD	A,A		; *4
	LD	HL,DispTbl	; get start of display table
	LD	C,A		; copy offset
	LD	B,#00h		; clear high byte
	ADD	HL,BC		; add offset
	LD	A,(HL)		; get width byte
	LD	(ChrWidt),A	; save new width
	LD	B,A		; copy width
	LD	A,(ChrHigh)	; get to chr height
	OR	B		; or in width nibble
	LD	B,A		; save size byte
	LD	A,#01h		; Set chr pitch instruction
	CALL	LCD_instr	; send display control instruction
	LD	A,B		; get size byte back
	OUT	(LCD_wrdata),A	; set mode
				; now set the other values
	INC	HL		; point to visible line length
	LD	A,(HL)		; get byte
	LD	(LineLen),A	; save it
	INC	HL		; point to offset low byte
	LD	C,(HL)		; get low byte
	INC	HL		; point to offset high byte
	LD	B,(HL)		; get high byte
	JP	Set_start	; set start address & return
;	RET

; Write chr to display. This routine will emulate a TTY display with auto
; scroll H (ON/OFF) and auto scroll V (ON/OFF) and a virtual display size
; of 64H x 32V characters.

; Control codes include....

;	0Ah	Line Feed
;	0Dh	Carrage Return
;	08h	Backspace
;	09h	Tab

LCD_print:
	PUSH	HL		; save HL
	LD	L,A		; save A for now
	LD	A,(DispMode)	; get mode
	AND	#02h		; check chr/graph mode
	JR	NZ,nextbit	; exit if graph mode

	LD	A,L		; get chr back
	AND	#7Fh		; clear bit 7
	CP	#20h		; compare with 20h
	JP	M,ctrl_chr	; jump if control character
	
				; normal chr so out to display
	LD	A,#0Ch		; write display data
	CALL	LCD_instr	; set instruction
	LD	A,L		; get chr back
	OUT	(LCD_wrdata),A	; chr out to display
				; now increment the cursor position
	PUSH	BC		; save BC
	LD	BC,(CursorAt)	; get cursor position
	INC	BC		; increment it
	JR	SetExit		; set cursor and exit

nextbit:
	POP	HL		; restore HL
	RET

ctrl_chr:			; control characters get here....
	PUSH	BC		; save BC
	LD	BC,(CursorAt)	; get cursor position
	LD	A,L		; get chr back
	CP	#0Dh		; was it CR
	JR	Z,WasCR		; jump if so

	CP	#0Ah		; was it LF
	JR	Z,WasLF		; jump if so

	CP	#09h		; was it TAB
	JR	Z,WasTA		; jump if so

	CP	#08h		; was it Backspace
	JR	Z,WasBA		; jump if so
	JR	CtrlDone	; else exit
SetExit:
	CALL	Set_cursor	; set the cursor
CtrlDone:			; gets here if not valid ctrl chr
	POP	BC		; restore BC
	POP	HL		; restore HL
	RET

WasCR:
	LD	A,#C0h		; set newline mask
	AND	C		; clear cursor x bits
	LD	C,A		; save new low byte
	JR	SetExit		; set cursor and exit

WasLF:
	LD	HL,0040h	; set linefeed offset
	ADD	HL,BC		; scroll to next line
	LD	C,L		; save new low byte
	LD	B,H		; save new high byte
	JR	SetExit		; set cursor and exit

WasTA:
	LD	A,C		; get low byte
	AND	#F8h		; mask out TAB bits
	ADD	A,#08h		; add TAB step
	LD	C,A		; save low byte
	JR	NC,SetExit	; if no carry set cursor and exit
	INC	B		; else increment high byte
	JR	SetExit		; set cursor and exit

WasBA:
	DEC	BC		; step back one
	JR	SetExit		; set cursor and exit

Msg:				; any old rubbish
	DB	'*   LCD Display  1234567890',09h,08h,'*',0Dh,0Ah
	DB	'*     32 x 8 chrs x lines',09h,08h,'*',0Dh,0Ah
	DB	0Ah
	DB	'*    Z80 driver software by',09h,08h,'*',0Dh,0Ah
	DB	'*',09h,'Lee Davison 1998',09h,08h,'*',0Dh,0Ah
	DB	0Ah
	DB	'ABCDEFGHIJKLMNOPQRSTUVWXYZ 01234',0Dh,0Ah
	DB	'abcdefghijklmnopqrstuvwxyz 56789'
MsgEnd:
Fix36:
	DB	'6'		; changes 32 to 36 for 8x7 chrs
Fix42:
	DB	'42'		; changes 36 to 42 for 8x6 chrs
FixEn:

TabSize:
	DB	08h		; display TAB step size
CursorAt:
	DW	0000h		; cursor address in display
DispMode:
	DB	10h		; display mode byte
				; (off/master/noblink/nocsr/cgr/intcg)
ChrHigh:
	DB	70h		; current character height (8 default)
ChrWidt:
	DB	07h		; current character width (8 default)
LineLen:
	DB	20h		; current visible line length (32 default)
StrtOff:
	DW	FFE0h		; default display start offset
DispOff
	DW	0000h		; offset for display
DispTbl:
	DB	05h,2Ah,EAh,FFh	; 6 bits/chr, 42 chrs vis, -16h offset
	DB	06h,24h,E4h,FFh	; 7 bits/chr, 36 chrs vis, -1Ch offset
	DB	07h,20h,E0h,FFh	; 8 bits/chr, 32 chrs vis, -20h offset
Graphics:
	DB	4		; 4 images
	DW	Pic1End-Pic1	; image size
Pic1:
	DB	A0h,FFh,01h,01h,9Eh,00h,02h,80h,01h,9Eh,00h,02h,80h,F9h,87h,FFh
	DB	01h,03h,8Eh,00h,01h,E0h,87h,FFh,02h,8Fh,F9h,87h,FFh,01h,07h,8Eh
	DB	00h,01h,E0h,87h,FFh,02h,9Fh,19h,87h,00h,01h,04h,8Eh,00h,0Eh,60h
	DB	00h,00h,10h,51h,DDh,FDh,FFh,9Fh,19h,00h,00h,C0h,03h,83h,00h,08h
	DB	04h,80h,FFh,80h,FFh,1Fh,E0h,3Fh,83h,00h,06h,FFh,01h,FCh,01h,60h
	DB	20h,85h,AAh,11h,EAh,9Fh,19h,00h,00h,C0h,03h,C0h,03h,00h,04h,E0h
	DB	00h,C2h,00h,00h,38h,83h,00h,28h,C0h,01h,00h,04h,00h,60h,00h,00h
	DB	40h,54h,75h,FFh,FFh,9Fh,19h,00h,00h,C0h,C3h,C3h,03h,C0h,07h,30h
	DB	FEh,03h,FEh,7Fh,8Ch,FFh,00h,00h,60h,FCh,07h,F6h,07h,60h,88h,88h
	DB	84h,AAh,03h,BBh,9Fh,B9h,87h,AAh,31h,06h,90h,FFh,0Fh,FFh,FFh,E4h
	DB	F0h,00h,00h,20h,87h,07h,F2h,03h,60h,00h,00h,11h,D1h,DDh,FDh,FFh
	DB	9Fh,D9h,03h,00h,C0h,C3h,C3h,03h,C0h,07h,D0h,83h,1Fh,0Fh,7Fh,34h
	DB	60h,00h,00h,A0h,01h,03h,DBh,01h,60h,00h,85h,AAh,05h,EEh,9Fh,D9h
	DB	03h,C0h,83h,C3h,1Bh,03h,C0h,07h,C0h,C9h,3Eh,FDh,33h,F4h,07h,7Eh
	DB	F8h,A0h,3Fh,80h,D9h,01h,60h,00h,00h,40h,55h,F5h,FFh,FFh,9Fh,D9h
	DB	87h,C3h,12h,07h,C0h,7Dh,BEh,0Dh,08h,34h,10h,00h,0Fh,A0h,81h,80h
	DB	DCh,01h,60h,88h,A8h,84h,AAh,03h,BBh,9Bh,B9h,87h,AAh,19h,06h,80h
	DB	3Fh,9Fh,E4h,0Fh,B6h,1Fh,F8h,E1h,B3h,FDh,C0h,D6h,01h,60h,00h,00h
	DB	10h,51h,DDh,FDh,FFh,9Fh,D9h,87h,C3h,11h,07h,00h,9Fh,8Fh,F6h,3Fh
	DB	F2h,71h,F0h,FCh,97h,8Fh,43h,D6h,01h,60h,20h,85h,AAh,03h,EAh,9Fh
	DB	D9h,87h,C3h,19h,07h,00h,CCh,03h,1Eh,78h,7Ah,F0h,E0h,3Fh,D7h,83h
	DB	67h,D3h,03h,60h,00h,00h,40h,54h,75h,FFh,FFh,9Fh,D9h,87h,C3h,12h
	DB	07h,00h,E6h,00h,06h,F8h,3Ah,D0h,C0h,9Fh,D1h,81h,36h,D3h,03h,60h
	DB	88h,88h,84h,AAh,03h,BBh,9Fh,F9h,87h,FFh,19h,07h,00h,33h,00h,00h
	DB	ECh,3Ah,D0h,81h,C7h,D0h,81h,9Eh,DFh,0Bh,60h,00h,00h,11h,D1h,DDh
	DB	FDh,FFh,9Fh,19h,87h,3Ch,11h,04h,80h,19h,06h,0Fh,E4h,3Bh,D0h,81h
	DB	7Fh,D0h,81h,DEh,C1h,19h,60h,00h,85h,AAh,03h,EEh,9Fh,19h,87h,3Ch
	DB	19h,04h,C0h,0Ch,83h,01h,F6h,3Bh,D8h,C1h,3Eh,D0h,C1h,CEh,FFh,1Fh
	DB	60h,00h,00h,40h,55h,F5h,FFh,FFh,9Fh,19h,86h,3Ch,13h,00h,04h,60h
	DB	06h,99h,FCh,F3h,3Ah,CCh,71h,3Eh,D0h,61h,CEh,FFh,1Fh,60h,88h,A8h
	DB	84h,AAh,03h,BBh,9Bh,B9h,87h,AAh,19h,06h,20h,FFh,8Dh,1Eh,F8h,F8h
	DB	E7h,19h,7Bh,C0h,3Fh,8Fh,FFh,0Fh,60h,00h,00h,10h,51h,DDh,FDh,FFh
	DB	9Fh,19h,86h,3Ch,12h,00h,04h,B0h,07h,0Ch,FEh,FFh,FCh,F0h,CDh,F1h
	DB	E0h,87h,0Fh,CEh,03h,60h,20h,85h,AAh,03h,EAh,9Fh,19h,83h,3Ch,36h
	DB	00h,3Ch,3Ch,00h,04h,90h,FFh,0Fh,FEh,7Fh,F0h,FFh,E4h,FCh,83h,FFh
	DB	07h,E2h,03h,60h,00h,00h,40h,54h,75h,FFh,FFh,9Fh,19h,00h,3Ch,3Ch
	DB	00h,3Ch,3Ch,00h,04h,C0h,FFh,1Fh,FCh,3Fh,F0h,FFh,70h,C0h,83h,FFh
	DB	07h,F8h,0Fh,60h,88h,88h,84h,AAh,03h,BBh,9Fh,B9h,87h,AAh,21h,06h
	DB	C0h,FFh,1Fh,F0h,0Fh,C0h,3Fh,F0h,F1h,0Fh,FEh,01h,F8h,0Fh,60h,00h
	DB	00h,11h,D1h,DDh,FDh,FFh,9Fh,19h,00h,3Ch,00h,00h,3Ch,3Ch,00h,04h
	DB	8Eh,00h,02h,60h,00h,85h,AAh,05h,EEh,9Fh,19h,00h,3Ch,83h,00h,03h
	DB	3Ch,00h,04h,8Eh,00h,0Ch,60h,00h,00h,40h,55h,F5h,FFh,FFh,9Fh,19h
	DB	00h,3Ch,83h,00h,03h,3Ch,00h,04h,8Eh,00h,03h,60h,88h,A8h,84h,AAh
	DB	03h,BBh,9Bh,19h,87h,00h,01h,04h,8Eh,00h,0Ah,60h,00h,00h,10h,51h
	DB	DDh,FDh,FFh,9Fh,F1h,87h,FFh,01h,07h,8Eh,00h,01h,C0h,87h,FFh,02h
	DB	9Fh,01h,9Eh,00h,02h,80h,01h,9Eh,00h,02h,80h,01h,9Eh,00h,02h,80h
	DB	01h,9Eh,00h,05h,80h,01h,00h,C0h,3Fh,87h,00h,01h,3Eh,8Fh,00h,01h
	DB	1Eh,83h,00h,06h,80h,01h,00h,70h,80h,01h,86h,00h,04h,03h,00h,80h
	DB	07h,8Ch,00h,01h,03h,83h,00h,05h,80h,01h,00h,18h,FFh,87h,00h,03h
	DB	F8h,00h,C0h,8Dh,00h,01h,78h,83h,00h,06h,80h,01h,00h,CCh,FFh,07h
	DB	86h,00h,04h,FCh,01h,00h,1Eh,8Ch,00h,01h,FCh,83h,00h,06h,80h,01h
	DB	00h,E4h,F1h,0Fh,86h,00h,04h,FCh,00h,00h,3Fh,8Ch,00h,01h,F8h,83h
	DB	00h,7Fh,80h,01h,00h,F6h,1Ch,F7h,EFh,C1h,7Fh,E0h,8Fh,03h,74h,3Eh
	DB	E0h,1Fh,F0h,03h,FEh,0Fh,00h,C0h,9Fh,C7h,03h,F0h,03h,F0h,E9h,E0h
	DB	1Fh,00h,80h,01h,00h,72h,C6h,11h,20h,60h,00h,20h,E0h,08h,F4h,03h
	DB	30h,00h,1Ch,08h,03h,00h,00h,40h,C0h,60h,08h,1Ch,08h,1Ch,ECh,30h
	DB	00h,00h,80h,01h,00h,FBh,71h,C0h,BFh,27h,FFh,81h,3Fh,0Eh,F4h,F8h
	DB	80h,7Fh,C6h,8Fh,F9h,3Fh,00h,00h,7Fh,3Eh,0Fh,C6h,0Fh,C6h,E7h,98h
	DB	7Fh,00h,80h,01h,00h,F9h,18h,C0h,DFh,8Fh,DFh,83h,9Fh,3Fh,74h,FEh
	DB	C1h,FFh,72h,3Fh,3Ch,78h,00h,00h,BFh,BFh,3Fh,72h,38h,72h,FEh,C8h
	DB	F9h,7Fh,00h,80h,01h,00h,3Dh,FEh,C3h,FFh,9Fh,43h,87h,FFh,7Fh,F4h
	DB	FFh,83h,3Fh,3Bh,71h,DEh,70h,00h,00h,DDh,FFh,7Fh,3Bh,78h,3Bh,F8h
	DB	ECh,ECh,01h,80h,01h,00h,3Dh,03h,40h,FFh,9Eh,7Fh,87h,7Eh,7Ch,F4h
	DB	E3h,83h,1Eh,19h,7Eh,7Ch,06h,00h,00h,7Dh,7Eh,7Ch,19h,68h,19h,F8h
	DB	64h,E7h,00h,80h,01h,00h,3Dh,F0h,4Fh,5Fh,DEh,70h,87h,1Eh,74h,F4h
	DB	A0h,83h,1Eh,1Dh,3Ch,F8h,03h,00h,00h,3Dh,3Ah,74h,1Dh,E8h,1Dh,E8h
	DB	F4h,71h,00h,80h,01h,00h,3Ch,F0h,47h,0Fh,67h,5Eh,87h,0Eh,74h,74h
	DB	A0h,83h,1Eh,1Dh,00h,C0h,1Fh,00h,00h,1Dh,3Ah,74h,1Dh,E8h,1Dh,E8h
	DB	F4h,7Fh,1Ch,00h,80h,01h,00h,7Eh,D0h,43h,0Fh,21h,4Fh,87h,0Eh,74h
	DB	74h,A0h,83h,1Eh,1Dh,90h,01h,3Eh,00h,00h,1Dh,3Ah,74h,1Dh,ECh,1Dh
	DB	ECh,74h,26h,00h,80h,01h,00h,FAh,DFh,43h,0Fh,A0h,77h,87h,3Eh,74h
	DB	74h,A0h,83h,1Eh,1Dh,9Ch,00h,7Bh,00h,00h,1Dh,3Ah,74h,1Dh,E6h,1Dh
	DB	E6h,F4h,33h,00h,80h,01h,00h,F8h,C0h,63h,0Fh,80h,1Fh,87h,0Eh,76h
	DB	74h,B0h,C3h,1Eh,FCh,C7h,FEh,79h,00h,80h,1Dh,3Ah,76h,FCh,F3h,FCh
	DB	F3h,F0h,9Fh,00h,80h,01h,00h,F8h,FFh,33h,1Fh,C0h,CFh,87h,EEh,73h
	DB	77h,98h,73h,3Eh,7Eh,70h,0Eh,7Ch,00h,C0h,1Ch,3Bh,73h,7Eh,F8h,7Eh
	DB	F8h,65h,F8h,C1h,01h,80h,01h,00h,F0h,FFh,97h,1Fh,00h,FFh,87h,FEh
	DB	B9h,F1h,C8h,17h,3Fh,F8h,7Fh,FEh,3Fh,00h,40h,3Eh,39h,79h,F8h,7Fh
	DB	F8h,FFh,E1h,FFh,01h,80h,01h,00h,E0h,FFh,CFh,7Fh,00h,FFh,8Fh,FEh
	DB	3Fh,FCh,E1h,CFh,FFh,F8h,3Fh,FCh,3Fh,00h,00h,7Fh,7Ch,FCh,F8h,7Fh
	DB	F8h,FFh,E3h,FFh,00h,80h,01h,00h,80h,7Fh,CEh,7Fh,00h,FCh,9Fh,EEh
	DB	0Fh,FEh,E1h,CFh,FFh,E0h,1Fh,F8h,0Fh,00h,00h,7Fh,7Ch,FCh,E0h,1Fh
	DB	E0h,FFh,C3h,7Fh,00h,80h,01h,87h,00h,02h,80h,0Eh,95h,00h,02h,80h
	DB	01h,87h,00h,02h,80h,0Eh,95h,00h,02h,80h,01h,87h,00h,02h,C0h,0Eh
	DB	95h,00h,02h,80h,01h,87h,00h,02h,60h,0Eh,95h,00h,02h,80h,01h,88h
	DB	00h,01h,1Fh,95h,00h,02h,80h,01h,87h,00h,02h,80h,3Fh,95h,00h,02h
	DB	80h,01h,9Eh,00h,02h,80h,01h,9Eh,00h,01h,80h,A0h,FFh
Pic1End:
	DW	Pic2End-Pic2	; image 2 size
Pic2:
	DB	55h,90h,20h,D2h,F6h,FFh,7Dh,ABh,27h,FEh,D7h,56h,80h,E2h,DEh,FBh
	DB	8Dh,62h,42h,50h,20h,49h,00h,00h,82h,00h,FEh,79h,05h,A8h,EFh,DFh
	DB	DEh,4Ah,94h,28h,D5h,5Eh,FBh,FAh,0Fh,AAh,F6h,0Dh,00h,58h,FBh,EEh
	DB	7Bh,1Ch,20h,40h,48h,00h,00h,40h,00h,00h,60h,96h,02h,50h,28h,25h
	DB	95h,00h,50h,42h,55h,FBh,D6h,EDh,AFh,FEh,BBh,36h,05h,90h,B7h,57h
	DB	AFh,69h,8Fh,C0h,90h,02h,85h,00h,1Bh,25h,00h,00h,55h,02h,86h,04h
	DB	05h,02h,A6h,56h,17h,FAh,DFh,2Ah,F0h,29h,00h,00h,7Ch,77h,D5h,09h
	DB	3Fh,00h,00h,09h,84h,00h,22h,40h,00h,00h,08h,00h,40h,85h,00h,AAh
	DB	52h,89h,95h,FAh,F4h,7Fh,03h,27h,57h,01h,00h,ACh,ADh,3Ah,0Dh,F8h
	DB	01h,80h,22h,01h,00h,00h,50h,00h,04h,84h,00h,1Ch,85h,40h,45h,09h
	DB	6Ah,6Bh,52h,D5h,DAh,27h,DAh,1Dh,40h,00h,58h,FBh,4Ah,19h,E0h,00h
	DB	C0h,04h,00h,48h,00h,40h,00h,48h,83h,00h,1Dh,40h,80h,11h,52h,8Ah
	DB	C8h,9Eh,56h,B5h,6Fh,C7h,D7h,F7h,8Bh,02h,B0h,56h,FDh,41h,00h,00h
	DB	F0h,2Bh,00h,30h,01h,55h,01h,90h,83h,00h,17h,08h,82h,C7h,A9h,54h
	DB	A2h,A5h,AAh,AAh,7Ah,DBh,75h,1Fh,B0h,04h,E0h,29h,03h,90h,C2h,A4h
	DB	BAh,07h,83h,00h,1Bh,05h,02h,84h,0Ah,00h,40h,D5h,80h,43h,14h,A5h
	DB	54h,6Bh,55h,5Dh,F7h,7Fh,DBh,24h,54h,01h,60h,7Bh,22h,58h,E1h,43h
	DB	84h,00h,05h,40h,0Ah,08h,00h,50h,83h,00h,14h,C1h,5Fh,50h,55h,A3h
	DB	55h,ADh,D2h,4Fh,C1h,BFh,1Fh,81h,00h,F0h,6Fh,0Ch,81h,FDh,17h,83h
	DB	00h,1Dh,90h,80h,0Ah,00h,00h,50h,01h,00h,80h,80h,0Dh,45h,29h,95h
	DB	5Dh,ABh,56h,3Dh,01h,6Fh,0Bh,20h,00h,58h,FDh,59h,A6h,FDh,29h,84h
	DB	00h,02h,80h,1Ah,83h,00h,16h,5Ah,55h,14h,81h,3Bh,AAh,D2h,AAh,B3h
	DB	54h,B5h,7Fh,44h,FAh,3Fh,00h,80h,FFh,3Fh,EBh,5Eh,0Fh,86h,00h,1Ah
	DB	F5h,12h,00h,00h,A0h,4Ah,65h,80h,EEh,80h,94h,1Bh,6Dh,ABh,EEh,FBh
	DB	89h,DAh,FDh,80h,80h,EDh,4Eh,64h,9Dh,ABh,86h,00h,1Dh,2Ah,EDh,1Fh
	DB	00h,00h,A8h,D6h,C0h,F7h,81h,AAh,FAh,AAh,DEh,F5h,FFh,23h,75h,B7h
	DB	5Fh,F0h,FFh,2Fh,A8h,E3h,22h,81h,00h,84h,83h,00h,1Ch,ACh,02h,A0h
	DB	80h,00h,00h,A8h,80h,7Fh,27h,A8h,95h,76h,B5h,FEh,FFh,5Fh,F1h,F7h
	DB	FFh,7Fh,BFh,07h,40h,B5h,94h,00h,10h,84h,00h,12h,28h,00h,04h,0Ah
	DB	80h,10h,00h,A0h,EFh,BFh,50h,B5h,E9h,EBh,FFh,FFh,77h,D0h,84h,FFh
	DB	7Fh,01h,02h,F5h,49h,02h,02h,00h,12h,02h,00h,50h,01h,00h,40h,28h
	DB	44h,29h,C0h,F7h,EDh,A5h,AAh,B2h,AFh,FBh,BFh,2Eh,70h,F7h,FEh,DFh
	DB	51h,01h,08h,1Eh,17h,02h,02h,FEh,43h,00h,00h,08h,0Ah,00h,00h,80h
	DB	10h,64h,D1h,EFh,FFh,4Bh,5Dh,FFh,0Bh,30h,7Eh,1Dh,F0h,FFh,EFh,7Eh
	DB	55h,00h,8Ah,ABh,B4h,04h,00h,3Fh,7Fh,02h,00h,48h,78h,ABh,54h,BBh
	DB	F7h,44h,D5h,FFh,BEh,B7h,D2h,B3h,73h,A0h,FAh,07h,F0h,ADh,FFh,7Fh
	DB	25h,00h,D0h,6Ah,C5h,82h,FFh,C7h,FFh,53h,04h,00h,80h,AAh,AAh,54h
	DB	00h,5Dh,B5h,FFh,FFh,5Fh,76h,54h,C0h,8Eh,F7h,00h,D0h,FEh,F7h,37h
	DB	0Ch,08h,00h,58h,49h,AAh,A3h,FAh,41h,FFh,4Fh,20h,04h,85h,00h,22h
	DB	20h,ECh,FFh,FFh,FBh,EFh,6Bh,05h,78h,1Ah,00h,D0h,FDh,BAh,FFh,07h
	DB	00h,17h,B4h,2Ah,ECh,7Ah,D7h,FFh,BFh,22h,0Ch,00h,20h,04h,00h,00h
	DB	47h,DDh,83h,FFh,10h,BBh,DEh,89h,40h,02h,00h,60h,BBh,F5h,00h,00h
	DB	49h,40h,60h,8Ah,F6h,83h,FFh,03h,7Fh,7Bh,05h,84h,00h,10h,A0h,10h
	DB	B6h,FFh,EDh,FFh,DFh,EAh,12h,55h,00h,00h,D0h,56h,03h,06h,85h,00h
	DB	0Eh,D1h,FFh,3Fh,FFh,FFh,CAh,3Eh,00h,05h,00h,00h,ADh,41h,DFh,83h
	DB	FFh,2Bh,5Fh,DDh,81h,AAh,00h,00h,50h,6Dh,48h,20h,02h,04h,00h,00h
	DB	02h,E8h,FFh,CFh,FFh,FFh,75h,25h,A2h,58h,55h,5Dh,49h,CAh,AAh,F8h
	DB	03h,00h,E0h,DAh,35h,14h,00h,00h,B4h,5Bh,00h,80h,01h,84h,00h,1Bh
	DB	C0h,FFh,FDh,FFh,AFh,7Bh,4Ah,2Dh,95h,AAh,02h,04h,B5h,CFh,02h,00h
	DB	00h,94h,FDh,F7h,6Bh,4Ah,A0h,62h,15h,45h,48h,85h,00h,1Ch,08h,FEh
	DB	FFh,7Fh,D5h,39h,55h,55h,A9h,04h,40h,40h,FEh,92h,00h,50h,FDh,BBh
	DB	ADh,7Fh,1Fh,80h,0Ah,A9h,03h,81h,2Dh,01h,84h,00h,0Eh,C4h,FDh,FFh
	DB	A2h,F6h,57h,AFh,AAh,CAh,A9h,00h,D0h,AAh,85h,85h,00h,09h,48h,E9h
	DB	7Fh,F5h,0Eh,40h,50h,91h,13h,83h,00h,0Bh,20h,B0h,F6h,FFh,EDh,76h
	DB	FFh,FFh,F3h,FFh,BFh,84h,FFh,84h,00h,0Ah,20h,55h,95h,A4h,AEh,0Bh
	DB	20h,20h,75h,03h,84h,00h,0Eh,08h,DBh,5Fh,77h,FBh,B6h,BFh,04h,FFh
	DB	D6h,FFh,7Fh,BBh,EAh,84h,00h,0Ah,48h,40h,75h,BBh,59h,0Dh,07h,A4h
	DB	6Ah,31h,83h,00h,0Fh,40h,40h,EDh,F7h,D7h,BEh,FBh,3Fh,07h,F8h,7Fh
	DB	ADh,D5h,EEh,9Dh,84h,00h,0Ah,80h,2Ah,4Ah,55h,B7h,07h,02h,48h,D5h
	DB	10h,83h,00h,0Fh,80h,A1h,D6h,0Dh,53h,5Fh,DFh,F7h,07h,D0h,55h,51h
	DB	0Ah,51h,E5h,84h,00h,12h,28h,AAh,5Ah,F7h,6Ah,0Dh,44h,B5h,BEh,5Fh
	DB	04h,00h,00h,20h,44h,16h,43h,D5h,83h,FFh,08h,83h,70h,ABh,82h,20h
	DB	08h,AAh,03h,83h,00h,4Ch,42h,55h,55h,ADh,56h,85h,40h,D0h,82h,BFh
	DB	F3h,81h,00h,90h,81h,B4h,16h,FEh,F6h,F5h,FFh,01h,DEh,2Ah,04h,00h
	DB	80h,A4h,16h,00h,00h,80h,ACh,AAh,AAh,6Ah,7Dh,3Bh,50h,ADh,8Dh,5Eh
	DB	85h,82h,00h,00h,40h,B3h,FEh,AFh,5Eh,FFh,FFh,80h,BAh,55h,01h,00h
	DB	00h,80h,40h,00h,00h,20h,ABh,52h,55h,55h,CBh,02h,C2h,1Ah,80h,FFh
	DB	56h,06h,83h,00h,09h,0Ch,FFh,FEh,EDh,FFh,3Fh,18h,6Dh,55h,83h,00h
	DB	01h,94h,83h,00h,1Dh,C0h,5Ah,A5h,AAh,AAh,BAh,81h,08h,27h,D8h,2Ch
	DB	00h,70h,00h,00h,20h,51h,F5h,B7h,F7h,F6h,0Fh,00h,D8h,52h,01h,00h
	DB	80h,A0h,83h,00h,03h,04h,55h,45h,83h,55h,05h,00h,A8h,04h,38h,35h
	DB	83h,00h,0Fh,06h,A0h,54h,B7h,FEh,DAh,FFh,03h,10h,B7h,96h,02h,00h
	DB	00h,80h,83h,00h,0Bh,A2h,D6h,4Ah,55h,D5h,2Ah,25h,82h,44h,16h,0Ah
	DB	83h,00h,1Dh,38h,00h,36h,ADh,EFh,F7h,FFh,33h,C4h,FAh,B5h,04h,00h
	DB	00h,80h,00h,05h,00h,A4h,AAh,51h,55h,B5h,2Dh,D0h,51h,00h,1Ah,2Ch
	DB	83h,00h,7Fh,90h,00h,00h,6Ah,5Bh,5Dh,F7h,2Ch,81h,AFh,2Ah,01h,00h
	DB	00h,80h,C3h,A8h,52h,ACh,B6h,4Ah,55h,55h,5Bh,22h,E9h,C4h,E7h,09h
	DB	00h,08h,20h,50h,00h,80h,D9h,F6h,EFh,2Bh,40h,60h,7Dh,ABh,0Ah,00h
	DB	00h,80h,7Fh,43h,84h,52h,A5h,92h,AAh,AAh,6Ah,90h,28h,60h,ADh,F3h
	DB	00h,00h,40h,78h,00h,00h,EEh,AFh,FAh,1Fh,88h,81h,FFh,5Fh,01h,00h
	DB	00h,DDh,AFh,54h,29h,25h,80h,54h,55h,55h,2Bh,C4h,02h,A8h,A5h,57h
	DB	01h,00h,20h,D6h,00h,40h,55h,7Dh,D7h,0Ah,00h,08h,EBh,FAh,17h,40h
	DB	FDh,81h,59h,90h,AAh,DAh,15h,A5h,AAh,AAh,3Ah,08h,84h,7Ah,9Bh,28h
	DB	28h,E0h,7Fh,01h,14h,00h,B4h,87h,ABh,7Eh,3Fh,40h,FEh,FFh,7Fh,04h
	DB	90h,FBh,BFh,65h,56h,D5h,56h,7Bh,49h,FDh,DDh,AAh,11h,05h,95h,0Ah
	DB	E0h,49h,B7h,47h,0Bh,10h,43h,0Ah,C0h,7Fh,FDh,E0h,BFh,BEh,DFh,05h
	DB	22h,D6h,B2h,D2h,5Bh,B7h,BAh,D6h,B6h,AAh,AAh,45h,80h,E0h,35h,14h
	DB	90h,D4h,8Fh,BCh,02h,80h,40h,45h,00h,60h,17h,65h,FFh,7Bh,6Dh,0Bh
	DB	54h,69h,88h,F4h,AEh,6Ah,D7h,BAh,2Dh,F5h,77h,80h,04h,54h,25h,20h
	DB	A8h,D3h,27h,16h,05h,80h,20h,01h,01h,8Ch,ABh,BAh,B5h,EDh,7Fh,95h
	DB	AAh,82h,82h,A9h,FBh,DDh,BAh,57h,DBh,AAh,AAh,50h,00h,B0h,2Eh,40h
	DB	A1h,50h,53h,DDh,BCh,26h,00h,A8h,02h,00h,88h,8Bh,EFh,FFh,DBh,AAh
	DB	2Ah,A9h,08h,80h,EBh,96h,52h,65h,FDh,B6h,DAh,7Dh,CFh,00h,4Ah,45h
	DB	48h,40h,A5h,96h,79h,0Eh,78h,60h,0Ah,00h,7Ch,FBh,7Dh,B5h,B6h,B7h
	DB	AAh,4Ah,01h,80h,AAh,6Bh,55h,ADh,AAh,2Dh,B5h,AAh,00h,08h,1Dh,01h
	DB	20h,2Ah,44h,55h,E7h,4Eh,14h,C3h,01h,00h,00h,F4h,EFh,FFh,DDh,DAh
	DB	55h,95h,02h,80h,D7h,6Ah,83h,55h,3Ch,FBh,A0h,A5h,C0h,42h,D1h,20h
	DB	84h,82h,10h,E8h,BAh,2Ah,CDh,D0h,1Ah,00h,D8h,7Dh,FFh,55h,EFh,6Dh
	DB	ABh,2Ah,00h,80h,ADh,D6h,AAh,2Ah,80h,56h,0Bh,15h,10h,6Dh,25h,40h
	DB	20h,40h,95h,12h,5Dh,EFh,00h,A4h,87h,B0h,30h,FFh,7Dh,BFh,BDh,B7h
	DB	ADh,A4h,00h,80h,4Bh,83h,55h,1Ch,00h,B4h,BAh,9Ah,E0h,90h,3Ah,1Eh
	DB	0Ch,80h,42h,A9h,B6h,3Ah,00h,11h,1Fh,44h,81h,B5h,FFh,6Bh,77h,7Dh
	DB	BBh,0Ah,01h,80h,83h,AAh,7Fh,2Ah,00h,F0h,A2h,02h,04h,57h,04h,80h
	DB	01h,00h,82h,48h,29h,EDh,80h,A0h,8Fh,19h,00h,FCh,F5h,EFh,DBh,EBh
	DB	56h,55h,00h,C0h,A1h,AAh,5Ah,15h,00h,40h,55h,09h,A0h,8Ah,0Ah,26h
	DB	04h,02h,88h,20h,73h,71h,31h,E8h,7Fh,02h,00h,AAh,AEh,5Fh,7Fh,5Fh
	DB	5Dh,95h,00h,80h,AAh,AAh,52h,15h,00h,A0h,AAh,40h,80h,55h,41h,00h
	DB	21h,00h,50h,01h,C4h,C0h,97h,F7h,FFh,00h,00h,78h,7Bh,F5h,D6h,B5h
	DB	ABh,12h,00h,80h,54h,29h,45h,1Dh,00h,20h,2Ah,20h,24h,49h,00h,00h
	DB	0Eh,00h,60h,03h,11h,12h,35h,FEh,7Fh,00h,00h,D4h,D6h,AFh,7Fh,6Fh
	DB	55h,05h,40h,A0h,ABh,52h,60h,0Ah,32h,00h,90h,10h,02h,54h,12h,02h
	DB	00h,09h,20h,C0h,00h,94h,D9h,D5h,FDh,2Dh,00h,00h,BCh,7Fh,FDh,D6h
	DB	ADh,2Ah,14h,00h,80h,AAh,A4h,50h,70h,00h,00h,02h,10h,9Ch,85h,00h
	DB	20h,50h,81h,80h,45h,C9h,80h,3Dh,6Ch,3Bh,00h,00h,FDh,AAh,ABh,7Fh
	DB	5Bh,D5h,AAh,10h,C2h,55h,95h,02h,80h,0Bh,00h,68h,80h,52h,40h,00h
	DB	48h,A0h,00h,22h,02h,AAh,53h,A9h,B7h,05h,00h,80h,ACh,7Fh,7Fh,BDh
	DB	AAh,36h,55h,25h,90h,55h,29h,84h,00h,1Dh,30h,80h,9Fh,11h,80h,02h
	DB	10h,80h,48h,09h,B8h,45h,74h,D5h,0Eh,00h,50h,FFh,EAh,AAh,F7h,55h
	DB	01h,00h,00h,E2h,2Ah,55h,09h,83h,00h,1Ch,A0h,51h,21h,08h,00h,00h
	DB	A8h,03h,42h,A2h,60h,53h,B5h,6Ah,01h,00h,A0h,FEh,DFh,7Fh,EDh,2Ah
	DB	14h,55h,01h,F4h,55h,02h,84h,00h,16h,80h,BAh,0Ah,4Ah,20h,04h,44h
	DB	01h,58h,4Ah,44h,50h,BBh,5Ch,00h,00h,0Eh,FFh,7Bh,D5h,5Bh,4Bh,83h
	DB	00h,04h,FDh,55h,95h,02h,84h,00h,1Bh,3Eh,0Eh,10h,0Ah,10h,F9h,02h
	DB	A0h,0Ah,FEh,A5h,44h,0Dh,00h,00h,DCh,FFh,FEh,6Fh,55h,17h,00h,00h
	DB	40h,FFh,A2h,22h,86h,00h,18h,89h,20h,00h,40h,52h,02h,00h,AAh,54h
	DB	AAh,2Ah,03h,01h,00h,E5h,BFh,FFh,5Ah,55h,34h,00h,00h,D0h,F7h
Pic2End:
	DW	Pic3End-Pic3	; image 3 size
Pic3:
	DB	A4h,FFh,02h,BFh,F7h,84h,FFh,02h,F7h,FDh,84h,FFh,10h,7Bh,7Fh,FDh
	DB	7Fh,ABh,DAh,ADh,DAh,BAh,DBh,D5h,FAh,6Eh,ADh,A5h,B6h,83h,FFh,1Dh
	DB	ADh,F5h,7Eh,6Dh,F7h,FFh,7Fh,ADh,56h,D5h,B7h,ADh,55h,ADh,FBh,7Eh
	DB	FFh,5Ch,55h,D5h,B2h,6Dh,B5h,AEh,86h,AAh,ABh,AAh,8Ah,83h,FFh,7Fh
	DB	FEh,5Eh,EBh,FFh,BFh,DBh,DFh,BDh,DFh,FFh,FDh,7Dh,FFh,F6h,3Eh,FBh
	DB	EFh,6Dh,EDh,36h,5Fh,DBh,6Eh,F9h,BAh,DBh,74h,11h,D5h,FFh,EFh,EFh
	DB	ABh,F7h,BDh,DBh,FEh,FEh,F6h,77h,75h,75h,5Fh,EFh,ADh,AFh,FFh,DEh
	DB	7Dh,D5h,56h,F5h,F5h,AAh,D9h,57h,6Bh,55h,ADh,EEh,D2h,BFh,7Dh,7Fh
	DB	FDh,DFh,DFh,FEh,EFh,FFh,ADh,DAh,DBh,EFh,F7h,BDh,FFh,75h,5Dh,79h
	DB	FFh,6Bh,BDh,57h,DBh,FDh,56h,DDh,BEh,6Dh,55h,91h,D2h,EFh,FFh,FFh
	DB	ADh,7Ah,F7h,D7h,7Dh,AFh,DAh,B7h,6Eh,F5h,DEh,EFh,6Ah,AFh,DFh,D7h
	DB	F7h,56h,EBh,ACh,B6h,55h,75h,B3h,6Bh,5Bh,EBh,4Ah,ADh,FFh,FFh,7Fh
	DB	7Fh,F7h,EFh,BDh,7Fh,DFh,FFh,BEh,EEh,75h,BFh,7Fh,BDh,FEh,5Ah,5Fh
	DB	EAh,BDh,B6h,AAh,F5h,55h,AFh,AEh,6Ah,DBh,B6h,94h,55h,A9h,FFh,DFh
	DB	DBh,BEh,FBh,FFh,FDh,F7h,D5h,6Dh,B7h,DFh,F5h,F7h,77h,BEh,6Fh,FDh
	DB	EDh,DFh,EDh,ADh,5Ah,7Dh,DDh,BAh,DDh,B6h,6Ah,6Bh,45h,D3h,7Fh,F7h
	DB	BFh,F7h,DEh,BEh,DFh,7Eh,5Fh,EFh,6Dh,F5h,EDh,DEh,1Eh,EEh,5Fh,AFh
	DB	DAh,F9h,55h,5Bh,B7h,D6h,B5h,ABh,AAh,D5h,D5h,5Ah,B5h,D4h,EFh,FFh
	DB	FDh,FEh,F7h,EBh,FBh,DFh,F5h,D9h,FFh,BFh,FEh,F7h,1Bh,F7h,6Dh,6Fh
	DB	ABh,F8h,F7h,6Ah,6Dh,ADh,6Ah,56h,5Dh,3Dh,5Bh,B5h,4Ah,A9h,FFh,7Fh
	DB	FBh,EFh,DFh,BEh,FFh,DEh,6Bh,ABh,7Eh,D5h,6Eh,B5h,BEh,0Fh,DEh,5Eh
	DB	D7h,6Eh,F8h,ADh,D5h,DBh,75h,DBh,ADh,ABh,EAh,AAh,AAh,5Ah,ABh,FFh
	DB	7Eh,FFh,FEh,EFh,B6h,FFh,DEh,7Eh,DBh,FFh,D5h,F5h,F7h,D2h,F7h,BBh
	DB	AEh,2Ah,BCh,55h,5Dh,55h,AFh,56h,7Bh,7Dh,ADh,75h,B5h,95h,D4h,FBh
	DB	FFh,DFh,DBh,FDh,FFh,DFh,FBh,DBh,7Fh,5Bh,7Fh,DBh,BEh,45h,FFh,6Eh
	DB	6Bh,35h,F8h,DBh,F5h,DAh,F4h,7Ah,A5h,52h,5Bh,CFh,6Ah,55h,D5h,DFh
	DB	FFh,7Bh,7Fh,B7h,EEh,7Bh,ADh,FEh,DEh,ADh,D5h,FDh,FBh,81h,DEh,4Fh
	DB	9Bh,CAh,EFh,37h,DFh,B7h,8Dh,A5h,DBh,DEh,AAh,BAh,ADh,AAh,AAh,83h
	DB	FFh,7Fh,EBh,FFh,7Bh,EDh,FFh,D7h,FBh,7Bh,5Dh,77h,03h,FAh,FFh,BEh
	DB	F6h,96h,BEh,D5h,BEh,7Ah,7Bh,5Bh,ADh,55h,DDh,56h,FBh,55h,D2h,FFh
	DB	7Fh,DFh,7Fh,F5h,DFh,FBh,6Eh,7Fh,57h,D5h,B7h,DEh,04h,C0h,DFh,E9h
	DB	ABh,0Ah,FAh,BFh,EBh,EFh,57h,55h,BBh,5Ah,A5h,E9h,AAh,CAh,ADh,FFh
	DB	EFh,FBh,EDh,DFh,FDh,AEh,FBh,F5h,FFh,BFh,EEh,15h,00h,00h,FFh,02h
	DB	DDh,F7h,FFh,EBh,BFh,5Ah,FFh,F6h,D6h,B7h,6Ah,95h,AAh,15h,D5h,FFh
	DB	BBh,FFh,7Fh,7Bh,F7h,FBh,6Fh,FFh,FEh,FFh,5Dh,4Fh,00h,00h,FEh,48h
	DB	50h,00h,78h,F7h,DDh,EFh,DFh,ADh,ADh,ECh,56h,77h,55h,55h,95h,FFh
	DB	FFh,7Fh,77h,FBh,EFh,DEh,6Eh,FBh,EFh,5Bh,EDh,FFh,16h,E0h,00h,FCh
	DB	21h,0Ah,00h,EAh,7Fh,BFh,BAh,7Dh,7Fh,7Bh,EBh,ADh,4Ah,ADh,AAh,AAh
	DB	7Fh,7Bh,FDh,7Fh,BDh,F7h,FBh,BFh,FDh,DFh,7Fh,FBh,05h,92h,0Eh,78h
	DB	81h,00h,00h,A8h,EFh,DFh,77h,FBh,EBh,AEh,DAh,DBh,B5h,5Ah,55h,D5h
	DB	FFh,DFh,DFh,EEh,FBh,5Eh,6Fh,FBh,7Fh,D8h,E3h,DFh,8Fh,82h,1Ch,FAh
	DB	28h,00h,00h,70h,BFh,7Eh,EDh,FFh,FFh,7Bh,D7h,B6h,AAh,B5h,55h,ADh
	DB	DFh,F6h,FEh,BBh,B7h,FBh,DBh,FFh,03h,C7h,3Fh,FFh,0Eh,FDh,39h,A8h
	DB	00h,00h,FCh,C5h,EFh,B7h,FBh,FFh,FFh,DFh,AAh,AFh,76h,6Bh,29h,A9h
	DB	FFh,13h,FFh,EFh,F6h,FDh,77h,FFh,7Fh,E0h,C1h,CBh,F1h,8Fh,FCh,37h
	DB	54h,08h,00h,4Bh,D6h,86h,FFh,3Ah,5Dh,5Fh,CDh,DAh,56h,D5h,FFh,DBh
	DB	BAh,BFh,D7h,DEh,F7h,2Fh,F0h,C2h,15h,84h,8Fh,FAh,5Bh,00h,00h,C0h
	DB	70h,DDh,BFh,FDh,FFh,FFh,FCh,F7h,B7h,BDh,BAh,AAh,ADh,AAh,DFh,FFh
	DB	EFh,FBh,7Eh,F7h,FDh,0Bh,7Ch,C0h,57h,18h,2Ch,75h,1Dh,00h,0Ah,50h
	DB	FDh,FBh,83h,FFh,7Fh,A8h,17h,BFh,DFh,AFh,AAh,56h,A9h,AAh,FFh,6Eh
	DB	BBh,EEh,EFh,DDh,FFh,02h,3Eh,80h,15h,60h,04h,CEh,06h,00h,00h,40h
	DB	FEh,DDh,FFh,FFh,87h,6Fh,6Fh,FCh,7Bh,F7h,AEh,6Ah,6Bh,D5h,EBh,FBh
	DB	FFh,BDh,7Ah,F7h,BFh,80h,0Fh,80h,57h,80h,85h,90h,01h,00h,0Ch,C8h
	DB	FEh,FBh,FFh,FFh,F4h,D5h,BFh,8Bh,FFh,BFh,FDh,AFh,9Ah,AAh,BFh,DEh
	DB	D6h,DBh,DFh,FDh,1Fh,C0h,17h,80h,0Bh,00h,91h,2Fh,08h,00h,51h,A1h
	DB	54h,FDh,FFh,0Fh,7Dh,B5h,3Fh,36h,7Ch,FFh,B7h,FFh,B3h,E9h,FBh,7Bh
	DB	BFh,F7h,7Ah,FFh,07h,C0h,0Bh,00h,57h,00h,00h,E8h,42h,80h,AAh,4Ah
	DB	EFh,FFh,FFh,D2h,35h,DFh,EAh,2Fh,ADh,F2h,FFh,FEh,7Dh,4Fh,96h,6Fh
	DB	EFh,DDh,DDh,DFh,FFh,05h,E0h,0Bh,00h,0Bh,01h,00h,90h,0Ah,92h,6Ah
	DB	15h,54h,EBh,5Fh,F5h,37h,D5h,7Fh,B8h,C4h,FFh,77h,D7h,BDh,F5h,FBh
	DB	BEh,EFh,BEh,F5h,7Fh,03h,E0h,05h,00h,37h,83h,00h,3Dh,20h,80h,FAh
	DB	AAh,B5h,DDh,57h,F9h,ADh,ECh,57h,60h,85h,F7h,FDh,FFh,77h,ABh,DFh
	DB	DBh,FDh,EBh,FFh,EFh,00h,F0h,11h,00h,56h,02h,00h,90h,84h,88h,5Fh
	DB	45h,02h,F1h,ABh,FEh,2Bh,F5h,7Fh,C0h,29h,BCh,7Fh,B7h,FEh,F5h,7Dh
	DB	7Fh,DFh,7Eh,EBh,5Fh,00h,B0h,02h,00h,0Eh,83h,00h,7Fh,08h,C0h,BBh
	DB	9Ah,00h,40h,ABh,7Fh,55h,FDh,2Dh,80h,0Bh,F0h,DFh,FFh,5Bh,AFh,D7h
	DB	DAh,F5h,EFh,FEh,3Bh,00h,78h,05h,00h,5Ch,04h,04h,80h,B2h,A0h,57h
	DB	35h,01h,80h,D6h,FFh,4Ah,F2h,1Fh,80h,15h,C0h,FFh,EEh,FFh,BEh,FFh
	DB	7Fh,DFh,BBh,FFh,0Fh,00h,58h,01h,00h,1Ch,89h,28h,01h,C0h,C0h,F5h
	DB	CAh,8Ah,00h,FFh,AFh,92h,FEh,16h,00h,ABh,80h,FBh,BBh,DAh,FFh,55h
	DB	B5h,75h,EDh,FDh,0Ah,00h,BCh,00h,00h,58h,20h,44h,05h,80h,C0h,FCh
	DB	BFh,2Ah,20h,FCh,BFh,AAh,FCh,3Fh,00h,07h,00h,FEh,EEh,7Fh,EBh,FFh
	DB	FFh,6Eh,B7h,FFh,07h,00h,2Ch,01h,00h,38h,00h,2Bh,7Fh,15h,84h,20h
	DB	FEh,D6h,B7h,A4h,FEh,6Bh,49h,BDh,35h,00h,ABh,00h,DCh,FFh,F7h,FFh
	DB	55h,ABh,DBh,FFh,FFh,02h,00h,5Eh,00h,00h,B8h,02h,00h,30h,00h,01h
	DB	EFh,FAh,FFh,8Bh,FEh,5Fh,92h,FEh,EFh,01h,16h,00h,F8h,EFh,DEh,FFh
	DB	FFh,7Eh,7Fh,D5h,BFh,03h,00h,56h,01h,00h,30h,00h,00h,B0h,00h,00h
	DB	9Bh,FFh,FEh,AAh,BFh,AAh,24h,FFh,5Eh,15h,56h,01h,F0h,FFh,F7h,EDh
	DB	DFh,EBh,F5h,FFh,FFh,00h,00h,5Fh,00h,00h,B0h,00h,00h,A0h,00h,00h
	DB	F2h,FFh,ABh,A5h,EFh,57h,49h,AFh,6Bh,55h,06h,04h,E0h,DFh,FFh,FFh
	DB	75h,BFh,DFh,D6h,5Fh,01h,00h,57h,00h,00h,30h,00h,57h,00h,B0h,00h
	DB	00h,DDh,BFh,FFh,FAh,FFh,2Ah,D2h,FFh,5Fh,55h,2Ch,01h,C0h,FEh,EEh
	DB	FEh,FFh,76h,75h,FFh,FFh,00h,00h,9Fh,00h,00h,B0h,00h,00h,C0h,00h
	DB	80h,FFh,FFh,AAh,FEh,DFh,4Bh,A5h,7Fh,B5h,AAh,4Ch,00h,C0h,FFh,BBh
	DB	FFh,ADh,EFh,EFh,FDh,57h,00h,80h,EFh,DFh,06h,60h,02h,00h,A0h,00h
	DB	04h,E8h,EFh,55h,FDh,7Bh,55h,E8h,DBh,6Fh,55h,ADh,02h,80h,FBh,FFh
	DB	FFh,F7h,BDh,5Dh,83h,FFh,17h,86h,FBh,FFh,BFh,61h,00h,FFh,CFh,01h
	DB	00h,EAh,BFh,6Fh,FFh,DFh,92h,D2h,FFh,5Ah,55h,09h,00h,80h,83h,FFh
	DB	2Ah,7Dh,F7h,F5h,7Dh,DBh,AEh,EBh,BEh,FDh,FBh,6Fh,FCh,FFh,FFh,00h
	DB	00h,90h,7Fh,59h,FFh,B7h,55h,F5h,EDh,B7h,52h,58h,02h,00h,DFh,FFh
	DB	FFh,EFh,BDh,AFh,BFh,B5h,FFh,FFh,EFh,F7h,DEh,83h,FFh,39h,7Fh,04h
	DB	A0h,52h,FFh,D7h,FFh,7Fh,25h,62h,BFh,ADh,54h,08h,08h,00h,FEh,FEh
	DB	EFh,77h,F7h,FEh,FEh,FFh,7Dh,BBh,FBh,BEh,F7h,FDh,EFh,FEh,7Fh,01h
	DB	08h,EAh,FFh,D7h,DFh,EEh,5Fh,F9h,FFh,56h,15h,58h,00h,00h,FEh,F7h
	DB	FBh,DBh,EDh,6Dh,6Fh,6Bh,EFh,83h,FFh,15h,7Fh,6Fh,FDh,DFh,1Fh,00h
	DB	00h,A8h,EFh,FFh,FFh,FDh,FFh,F2h,B5h,FFh,03h,18h,01h,00h,6Ch,83h
	DB	FFh,72h,BFh,FBh,FFh,BFh,7Dh,FFh,BFh,EDh,DDh,FFh,5Fh,FBh,0Fh,00h
	DB	00h,A0h,FEh,FFh,FFh,BFh,B6h,7Fh,DFh,DDh,EEh,29h,10h,00h,FCh,77h
	DB	FFh,B5h,FBh,DEh,DFh,F6h,EFh,B5h,EAh,FFh,FFh,FBh,FFh,FFh,07h,00h
	DB	00h,54h,FFh,FFh,7Fh,FBh,FFh,DDh,FDh,B6h,BBh,57h,BFh,05h,F8h,DDh
	DB	FFh,FFh,DEh,FBh,F5h,BFh,FFh,FFh,7Fh,EFh,77h,7Fh,77h,FFh,83h,00h
	DB	00h,A0h,FAh,FFh,FFh,DFh,6Eh,7Fh,B7h,DFh,FFh,FDh,F5h,FFh,F9h,F7h
	DB	FDh,DFh,7Bh,6Fh,DFh,F6h,DBh,EEh,EEh,FFh,DEh,F7h,FDh,FBh,01h,00h
	DB	80h,AAh,FFh,EFh,83h,FFh,17h,EDh,DDh,7Ah,BBh,5Fh,FFh,55h,BFh,BFh
	DB	DFh,FFh,FEh,FDh,FBh,AFh,FFh,BFh,FFh,B7h,FFh,FDh,DFh,FEh,83h,00h
	DB	0Ch,54h,FDh,FFh,FBh,77h,DBh,7Fh,F7h,EFh,DFh,FDh,5Bh,83h,FFh,0Eh
	DB	F7h,FFh,DFh,AFh,6Fh,FFh,EEh,FBh,BBh,FDh,7Fh,BFh,FFh,3Fh,83h,00h
	DB	10h,40h,FBh,FFh,DFh,BFh,6Fh,EFh,5Dh,7Bh,75h,FFh,FEh,DDh,FDh,EBh
	DB	FEh,85h,FFh,08h,7Fh,FFh,FDh,BFh,EDh,FFh,DDh,3Fh,83h,00h,0Fh,48h
	DB	EEh,FFh,F6h,F7h,FDh,7Dh,EFh,DDh,FFh,B7h,FFh,F7h,EFh,FEh,83h,FFh
	DB	01h,FDh,83h,FFh,07h,DBh,EFh,EDh,FFh,76h,F7h,3Ah,83h,00h,10h,92h
	DB	F5h,BFh,FFh,FFh,DFh,D7h,DDh,F7h,17h,F8h,EBh,BDh,FBh,FFh,AAh,83h
	DB	FFh,0Ah,FBh,DFh,FFh,FFh,7Eh,7Fh,DFh,FFh,BFh,0Fh,83h,00h,10h,20h
	DB	FDh,FFh,FFh,BFh,F7h,7Ah,BFh,5Dh,A8h,BFh,7Eh,FFh,FEh,B7h,DDh,85h
	DB	FFh,08h,7Fh,FFh,FFh,F7h,FDh,FFh,FDh,03h,83h,00h,02h,A4h,FAh,83h
	DB	FFh,0Bh,DEh,DFh,F7h,B7h,45h,FFh,DFh,EFh,F7h,FEh,EBh,87h,FFh,06h
	DB	FBh,FFh,FFh,BFh,EFh,02h,83h,00h,02h,49h,EDh,83h,FFh,0Bh,F7h,FAh
	DB	DEh,7Fh,D5h,D5h,F5h,FDh,BFh,EFh,D6h,85h,FFh,07h,EFh,FFh,7Eh,FFh
	DB	77h,FBh,55h,84h,00h,03h,52h,FBh,FBh,83h,FFh,0Ah,AFh,B7h,B6h,E5h
	DB	FFh,7Fh,BFh,F6h,7Fh,FDh,86h,FFh,06h,EEh,EFh,BBh,FEh,4Fh,2Ah,85h
	DB	00h,01h,6Ah,86h,FFh,08h,FBh,BBh,6Eh,EFh,EBh,7Fh,D5h,EBh,87h,FFh
	DB	05h,BFh,FFh,7Fh,91h,02h,84h,00h,04h,48h,D5h,FFh,F7h,86h,FFh,06h
	DB	FBh,FDh,FFh,DDh,B6h,B6h,83h,FFh,09h,CFh,F7h,FFh,F7h,FDh,FFh,57h
	DB	45h,24h,84h,00h,04h,22h,AAh,FEh,DFh,88h,FFh,04h,7Eh,FFh,EDh,EEh
Pic3End:
	DW	Pic4End-Pic4	; image 4 size
Pic4:
	DB	05h,ADh,ADh,7Fh,7Dh,81h,84h,00h,09h,A0h,A0h,AAh,2Ah,2Dh,6Ah,55h
	DB	A5h,0Ah,84h,00h,10h,40h,84h,68h,6Bh,1Dh,DBh,F6h,77h,B7h,9Eh,7Fh
	DB	FFh,DBh,2Ah,22h,02h,83h,00h,09h,0Eh,AAh,B5h,FEh,CAh,B9h,AAh,4Eh
	DB	05h,85h,00h,0Eh,20h,A0h,8Ah,7Ah,BDh,DBh,A9h,5Eh,F7h,DBh,EBh,FEh
	DB	7Fh,11h,84h,00h,0Ch,39h,ACh,6Eh,95h,DEh,A2h,6Ah,A9h,02h,00h,00h
	DB	82h,83h,00h,0Dh,40h,B7h,EFh,D7h,FFh,6Ah,E5h,EBh,EFh,BFh,F7h,56h
	DB	08h,84h,00h,0Ch,85h,A8h,5Ah,53h,85h,97h,9Ah,4Bh,0Dh,00h,40h,08h
	DB	83h,00h,0Dh,90h,6Ch,5Dh,55h,55h,A8h,62h,C5h,7Dh,FDh,FDh,EFh,02h
	DB	84h,00h,0Ch,22h,AAh,95h,15h,69h,6Dh,65h,F5h,04h,00h,82h,42h,84h
	DB	00h,0Bh,49h,E3h,55h,8Dh,5Dh,90h,90h,F7h,D7h,5Fh,ABh,85h,00h,0Ch
	DB	5Ah,55h,75h,55h,B7h,B8h,5Bh,6Bh,15h,80h,10h,10h,83h,00h,0Dh,40h
	DB	15h,05h,94h,22h,34h,85h,D0h,DEh,FEh,FEh,02h,05h,83h,00h,0Ah,80h
	DB	AAh,B2h,96h,24h,FFh,52h,64h,FBh,57h,83h,00h,01h,01h,83h,00h,0Ch
	DB	92h,08h,2Bh,20h,23h,81h,CBh,BBh,DBh,DBh,0Ah,20h,84h,00h,0Ch,6Bh
	DB	6Ah,79h,B9h,85h,AEh,9Dh,AAh,54h,3Dh,92h,02h,84h,00h,0Ch,20h,B0h
	DB	14h,F4h,45h,40h,DCh,EFh,7Fh,57h,A5h,40h,84h,00h,0Dh,EAh,9Ah,4Eh
	DB	A3h,BFh,DAh,6Fh,EBh,5Fh,D5h,02h,40h,02h,83h,00h,0Ch,88h,4Ah,8Bh
	DB	28h,5Fh,01h,84h,BAh,AAh,ADh,4Ah,01h,84h,00h,0Ch,69h,EAh,BBh,5Ah
	DB	40h,55h,95h,76h,B5h,75h,07h,08h,85h,00h,0Bh,D4h,04h,CAh,12h,B0h
	DB	A9h,FDh,77h,ABh,2Ah,15h,83h,00h,0Dh,40h,66h,6Eh,A5h,B4h,9Bh,2Ah
	DB	55h,EBh,D2h,D5h,3Dh,80h,85h,00h,0Bh,50h,15h,B0h,DAh,26h,C9h,B1h
	DB	DAh,5Ah,A5h,04h,83h,00h,0Dh,10h,BDh,5Bh,2Ah,DAh,54h,59h,55h,5Fh
	DB	2Fh,6Eh,D6h,20h,86h,00h,1Ah,40h,85h,4Ah,DDh,8Ah,EFh,57h,ABh,92h
	DB	A8h,02h,00h,00h,6Dh,A9h,D6h,56h,55h,ADh,46h,EAh,5Bh,35h,91h,6Dh
	DB	07h,86h,00h,1Ah,08h,D2h,A2h,ABh,F2h,77h,DDh,2Ah,49h,05h,40h,04h
	DB	50h,12h,A2h,94h,94h,6Ah,77h,59h,BFh,FEh,ABh,7Fh,DBh,1Ah,86h,00h
	DB	1Ah,A0h,2Ch,64h,95h,93h,DFh,6Bh,55h,12h,12h,01h,80h,BCh,ABh,54h
	DB	B5h,2Ah,57h,CDh,52h,D5h,F7h,5Fh,54h,BDh,36h,86h,00h,1Ah,40h,D1h
	DB	AAh,24h,A2h,7Dh,5Dh,A5h,44h,01h,00h,00h,A9h,CAh,AFh,6Ah,D6h,59h
	DB	6Dh,8Bh,FEh,5Dh,F5h,B5h,EBh,BDh,86h,00h,1Bh,80h,94h,54h,8Dh,CAh
	DB	EFh,ABh,0Ah,11h,24h,00h,C0h,EAh,12h,59h,D5h,15h,ABh,32h,B1h,B5h
	DB	F7h,BFh,ABh,5Ah,A5h,02h,85h,00h,7Fh,20h,2Ah,AFh,22h,D0h,BFh,AAh
	DB	AAh,44h,01h,00h,90h,35h,B7h,EFh,58h,D5h,B6h,6Ch,A5h,DFh,BEh,8Bh
	DB	BFh,6Ah,DBh,04h,00h,08h,04h,01h,00h,00h,A5h,6Ah,C9h,D5h,DBh,56h
	DB	11h,02h,00h,00h,A0h,D4h,E9h,B4h,A6h,2Bh,49h,BBh,52h,F5h,EDh,3Eh
	DB	68h,DBh,05h,AAh,00h,6Bh,5Fh,02h,00h,00h,40h,57h,45h,FFh,6Fh,55h
	DB	4Ah,51h,04h,00h,88h,5Dh,DAh,ABh,A9h,76h,ADh,BCh,95h,7Eh,BBh,0Fh
	DB	40h,A0h,77h,55h,BEh,A4h,A2h,02h,00h,00h,44h,BFh,D5h,FFh,DDh,AAh
	DB	10h,84h,00h,00h,F5h,66h,BFh,AEh,B5h,ABh,55h,AFh,AAh,DAh,F6h,03h
	DB	40h,57h,55h,A9h,F6h,FFh,6Bh,05h,7Fh,00h,00h,D0h,4Eh,59h,D9h,ABh
	DB	2Ah,45h,11h,00h,80h,1Ah,05h,A0h,39h,50h,B6h,AAh,AAh,69h,75h,ABh
	DB	00h,A0h,ACh,FEh,FFh,AFh,0Dh,ABh,12h,00h,00h,80h,2Dh,EFh,E6h,B6h
	DB	8Ah,0Ah,24h,00h,40h,25h,A2h,5Bh,06h,D4h,D6h,FFh,5Bh,A7h,B6h,3Eh
	DB	00h,58h,FDh,DFh,AAh,64h,51h,6Bh,05h,00h,00h,50h,AAh,5Dh,BBh,ADh
	DB	10h,50h,01h,00h,50h,4Bh,29h,A4h,00h,50h,2Bh,A8h,76h,9Ah,EDh,1Bh
	DB	00h,D0h,16h,A9h,6Ah,56h,0Bh,ABh,2Ah,00h,00h,80h,D2h,B6h,FFh,2Bh
	DB	05h,00h,48h,00h,B0h,A6h,DAh,2Ah,00h,6Ch,D5h,AAh,C5h,1Dh,55h,15h
	DB	80h,AAh,DFh,74h,53h,59h,ADh,A7h,7Fh,94h,00h,00h,A0h,AAh,F6h,85h
	DB	56h,09h,00h,02h,00h,5Ch,61h,AAh,2Ah,00h,ABh,ABh,EAh,5Ah,F3h,52h
	DB	33h,77h,F5h,A5h,A6h,F2h,46h,97h,ABh,85h,00h,00h,A8h,AAh,BFh,FFh
	DB	2Dh,00h,80h,20h,80h,92h,56h,ADh,0Dh,00h,F6h,54h,95h,52h,8Ah,ABh
	DB	7Eh,A9h,7Dh,2Dh,BDh,BEh,FBh,E5h,ADh,2Ah,00h,00h,50h,55h,AAh,CCh
	DB	95h,02h,20h,5Ah,55h,B6h,29h,57h,03h,40h,29h,55h,FDh,FFh,6Ah,7Eh
	DB	A9h,D6h,AFh,EAh,CAh,2Dh,4Ah,68h,55h,95h,00h,00h,54h,A9h,EAh,C1h
	DB	22h,06h,90h,A5h,DAh,FCh,AEh,F8h,01h,B0h,BEh,D5h,DFh,AEh,ABh,E9h
	DB	EFh,7Fh,5Bh,5Dh,ADh,D9h,7Dh,B2h,7Fh,76h,13h,01h,00h,A0h,68h,4Bh
	DB	BEh,15h,29h,04h,FAh,FFh,77h,B3h,0Fh,A0h,54h,A7h,FAh,B7h,B1h,2Eh
	DB	B7h,7Eh,57h,D5h,B2h,6Dh,55h,ACh,45h,49h,ABh,00h,00h,12h,D1h,96h
	DB	A8h,02h,CBh,FBh,AAh,52h,A5h,5Dh,D2h,2Ah,D5h,55h,97h,64h,AAh,58h
	DB	E9h,EAh,A6h,1Ah,57h,7Ah,2Dh,FBh,AFh,96h,2Ah,01h,00h,40h,A4h,EEh
	DB	FFh,08h,61h,94h,5Ah,56h,DAh,F5h,6Ah,5Fh,FDh,AAh,E9h,CBh,5Eh,50h
	DB	56h,8Bh,59h,55h,29h,ADh,EBh,B5h,B4h,D6h,55h,01h,00h,00h,40h,BDh
	DB	FFh,01h,53h,89h,45h,BBh,ADh,4Ah,BFh,FAh,2Fh,E5h,BEh,3Eh,55h,B5h
	DB	D5h,FAh,FAh,FFh,46h,AFh,FBh,50h,23h,B5h,B7h,AAh,00h,00h,A8h,A4h
	DB	FBh,FFh,22h,72h,95h,B0h,45h,E7h,B5h,B2h,F7h,BAh,EAh,7Ah,65h,B5h
	DB	5Ah,D5h,65h,5Fh,55h,D1h,BAh,02h,BFh,CAh,E5h,5Bh,83h,00h,3Dh,10h
	DB	44h,FDh,08h,6Dh,7Ah,EDh,B6h,5Ah,6Bh,EDh,9Ah,D5h,B4h,4Ah,BDh,ADh
	DB	6Ch,B5h,ABh,57h,ABh,B2h,52h,DFh,BAh,F4h,AEh,56h,01h,00h,08h,80h
	DB	0Ah,EBh,12h,62h,A5h,BDh,B5h,7Ah,D5h,54h,F3h,AAh,BEh,55h,ADh,2Dh
	DB	BFh,AAh,6Eh,D1h,A6h,FCh,55h,EDh,B7h,53h,71h,AFh,84h,00h,1Dh,40h
	DB	D4h,00h,55h,12h,69h,A2h,A4h,DAh,EAh,1Ah,DBh,5Ah,D7h,57h,D6h,A9h
	DB	94h,B9h,5Bh,A9h,7Fh,77h,71h,4Dh,56h,DDh,2Ah,01h,84h,00h,1Ch,A0h
	DB	0Ah,D1h,4Ch,85h,9Ch,9Bh,D3h,BFh,AFh,56h,59h,6Ah,D4h,D5h,DAh,EAh
	DB	B7h,87h,FCh,ABh,5Ch,BFh,EDh,DFh,6Ah,55h,03h,84h,00h,1Ch,80h,00h
	DB	ABh,C2h,75h,FBh,2Dh,BDh,6Ah,D5h,B6h,96h,75h,15h,F5h,25h,D7h,CCh
	DB	FAh,BFh,56h,31h,45h,B5h,38h,D7h,76h,02h,84h,00h,1Ch,80h,12h,42h
	DB	74h,FFh,B5h,EAh,E0h,9Ah,B1h,AFh,B8h,BAh,D5h,BEh,B5h,E4h,BDh,D2h
	DB	D7h,B5h,6Eh,BBh,FAh,A2h,BCh,A5h,02h,84h,00h,1Bh,80h,00h,51h,5Dh
	DB	D1h,5Eh,56h,1Fh,36h,FEh,B5h,6Bh,5Ah,ADh,56h,5Ah,75h,C3h,FEh,B7h
	DB	AAh,CAh,DFh,5Dh,6Fh,67h,A7h,85h,00h,1Ch,80h,08h,55h,53h,65h,82h
	DB	CAh,ADh,ADh,5Dh,DBh,76h,6Bh,53h,95h,AEh,7Ch,7Ah,FFh,FFh,57h,FBh
	DB	6Ah,6Fh,FDh,ADh,55h,01h,84h,00h,1Ch,80h,40h,F4h,B5h,58h,EBh,F4h
	DB	EAh,FFh,B7h,B6h,A4h,A6h,ECh,A9h,4Bh,EDh,FDh,DBh,FFh,FEh,FEh,B6h
	DB	DBh,56h,D5h,2Ah,01h,84h,00h,1Ch,80h,00h,51h,4Dh,E9h,43h,BFh,5Ah
	DB	6Dh,83h,AAh,53h,FDh,BFh,7Fh,5Ah,B7h,FFh,FFh,F6h,F7h,FFh,4Bh,6Dh
	DB	55h,B5h,4Bh,01h,84h,00h,1Ch,80h,00h,6Ah,D1h,38h,6Dh,43h,A5h,8Dh
	DB	AEh,9Dh,B5h,62h,65h,AAh,94h,57h,DDh,FFh,DFh,7Fh,0Bh,B4h,AAh,BAh
	DB	6Eh,2Dh,01h,84h,00h,7Fh,DAh,00h,69h,6Dh,67h,C9h,D5h,6Eh,B2h,B5h
	DB	D6h,ADh,ADh,4Ah,C7h,77h,ADh,FFh,FFh,7Fh,FFh,AAh,A8h,D5h,A6h,F5h
	DB	4Fh,02h,00h,20h,DBh,E9h,AAh,00h,55h,BDh,59h,35h,ABh,A9h,56h,DDh
	DB	ADh,76h,69h,54h,35h,BEh,FAh,FFh,FBh,FFh,2Ah,55h,D6h,FFh,FFh,7Fh
	DB	5Fh,05h,00h,50h,25h,5Ah,B9h,00h,ABh,1Ah,EEh,FCh,A2h,AFh,EAh,FBh
	DB	F7h,EAh,17h,69h,E5h,D7h,EAh,FFh,BFh,FFh,F7h,FFh,FBh,FFh,FEh,DEh
	DB	BFh,04h,00h,2Ah,BAh,9Dh,E5h,00h,DAh,E2h,5Bh,97h,6Ah,A5h,FCh,FFh
	DB	BFh,AEh,FDh,FFh,FFh,BAh,7Dh,FFh,FFh,6Dh,DFh,FFh,7Fh,AAh,ABh,FBh
	DB	EDh,05h,A0h,DDh,25h,12h,2Dh,9Fh,00h,55h,5Dh,E5h,6Bh,7Dh,55h,DFh
	DB	FFh,FFh,55h,6Bh,ABh,AAh,AAh,FAh,83h,FFh,7Fh,FBh,F7h,BFh,7Fh,FFh
	DB	6Fh,7Fh,01h,D0h,6Bh,FCh,53h,ACh,00h,59h,B7h,5Ah,AEh,82h,94h,F7h
	DB	DFh,F6h,DBh,57h,1Dh,ABh,6Ah,DDh,FFh,EEh,00h,60h,BDh,D6h,EFh,6Ah
	DB	BDh,ABh,03h,6Ah,11h,2Dh,C8h,FAh,00h,F5h,5Dh,D5h,51h,81h,EAh,FDh
	DB	FDh,FFh,3Bh,DDh,D5h,5Ah,7Dh,FBh,7Fh,1Bh,00h,78h,3Bh,6Bh,BBh,FFh
	DB	F7h,FFh,07h,D1h,56h,ACh,37h,95h,04h,15h,55h,77h,4Fh,B6h,A2h,EFh
	DB	EFh,DFh,D6h,37h,65h,EAh,C2h,FEh,DFh,03h,00h,52h,57h,FBh,FEh,EDh
	DB	3Fh,00h,00h,16h,87h,FFh,4Dh,EBh,02h,D5h,AAh,DAh,30h,FDh,FFh,FEh
	DB	FFh,76h,A8h,EAh,F6h,55h,BDh,D5h,FFh,01h,05h,00h,BDh,DAh,4Eh,1Fh
	DB	84h,00h,1Ch,D9h,E8h,56h,ABh,BAh,00h,95h,FAh,FFh,DFh,B4h,5Fh,FBh
	DB	DDh,03h,58h,6Fh,5Dh,B7h,6Bh,EDh,E7h,00h,00h,D6h,ADh,7Ah,03h,83h
	DB	00h,1Dh,80h,E6h,9Bh,07h,55h,ABh,15h,EEh,6Dh,4Bh,F7h,FFh,FAh,5Eh
	DB	F7h,00h,54h,ABh,BEh,ACh,B6h,5Bh,7Bh,00h,A0h,57h,6Bh,D7h,01h,84h
	DB	00h,1Bh,DDh,09h,F9h,75h,F2h,22h,50h,DBh,7Dh,DDh,DDh,5Fh,7Bh,3Dh
	DB	00h,BAh,BDh,A6h,EAh,5Dh,75h,57h,00h,C0h,EAh,AEh,ADh,84h,00h,1Ch
	DB	80h,50h,74h,77h,ADh,DEh,95h,80h,56h,DBh,7Bh,FBh,EBh,D6h,1Fh,00h
	DB	56h,BBh,5Bh,5Dh,F7h,ADh,75h,00h,7Ch,55h,EBh,76h,84h,00h,1Ch,40h
	DB	7Dh,AFh,F5h,D7h,F6h,2Ah,12h,6Ch,2Dh,95h,40h,B6h,35h,0Bh,80h,D5h
	DB	6Fh,2Dh,53h,ABh,76h,DBh,00h,8Dh,EAh,BCh,2Dh,85h,00h,1Bh,62h,DCh
	DB	FFh,B7h,FDh,95h,40h,B0h,FAh,5Bh,AAh,22h,EDh,02h,80h,EAh,5Ah,57h
	DB	B5h,D7h,DDh,AAh,A2h,A9h,5Ah,D7h,16h,84h,00h,1Dh,10h,58h,7Fh,FDh
	DB	7Fh,D7h,2Bh,01h,81h,ABh,3Eh,40h,D9h,DAh,04h,B4h,7Ah,EFh,DAh,6Ch
	DB	3Fh,EDh,FAh,6Bh,75h,ADh,76h,07h,40h,83h,00h,28h,20h,F0h,DBh,F5h
	DB	F7h,BFh,6Eh,22h,02h,00h,00h,10h,B2h,AAh,9Bh,54h,AEh,B5h,B6h,D2h
	DB	D5h,AAh,95h,5Fh,AEh,B6h,DDh,02h,08h,01h,00h,00h,24h,67h,DFh,FFh
	DB	FFh,D6h,9Bh,84h,83h,00h,3Bh,84h,A5h,1Ah,56h,CBh,D5h,6Dh,ADh,DAh
	DB	6Dh,D3h,76h,ADh,A5h,E5h,96h,01h,42h,04h,00h,00h,40h,D8h,6Ah,59h
	DB	FFh,F5h,6Fh,55h,24h,80h,04h,00h,A8h,A6h,B5h,76h,69h,DAh,D5h,F5h
	DB	DAh,27h,5Ah,F7h,6Bh,BDh,6Dh,00h,00h,21h,00h,00h,90h,FCh,FFh,7Fh
	DB	7Dh,ADh
Pic4End:
	END
